/**
 * \file: signature.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#ifndef ALD_ENCRYPTION_SDC_ACCESS_H
#define ALD_ENCRYPTION_SDC_ACCESS_H

#include "ald_types.h"
#include "model/daemon_model.h"

//---------------------------------------- init and deinit ----------------------------------------------------------
/// TODO: Do you need something being done once per running instance of the ald??
///      --> uncomment, implement, and connect these functions
// error_code_t sdc_access_init(void);
// void sdc_access_deinit(void);
//-------------------------------------------------------------------------------------------------------------------

//---------------------------------------- reading and writing the state file ---------------------------------------
/**
 * loads and decrypts the content of the ALD state file that stores the current level.
 * \param fn the absolute path to the encrypted file
 * \param buf the buffer in which the decrypted content of the file is to be stored
 * \param buf_size the size of the buffer
 * \return returns RESULT_OK in case the file could be read out and decrypted successfully
 * 		- return RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES in case the file could not be read at all.
 *		- return RESULT_WRAPPED_FILE_CORRUPT in case the resulting data does not match buf_size.
 *		- return RESULT_CRYPTOGRAPHIC_OPERATION_FAILED in case the content of the file could not be decrypted successfully
 *		- return RESULT_NORESOURCE in case the resources are not available
 */
error_code_t sdc_access_load_state_file(const char *fn, void *buf, size_t buf_size);

/**
 * Stores and encrypts the data in buf (of size buf_size) into the ALD state file.
 * \param fn the absolute path to the encrypted file
 * \param buf the buffer which contains the data to be written into the state file. This buffer must not be changed
 * within this function.
 * \param buf_size the size of the buffer
 * \return returns RESULT_OK in case the file could be encrypted and written successfully
 * 		- return RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES in case of problems opening or accessing the file
 *		- return RESULT_CRYPTOGRAPHIC_OPERATION_FAILED in case the data could not be encrypted successfully
 *		- return RESULT_NORESOURCE in case the resources are not available
 */
error_code_t sdc_access_store_state_file(const char *fn, const void *buf, size_t buf_size);
//-------------------------------------------------------------------------------------------------------------------

//---------------------------------------- decrypting a block in memory ---------------------------------------------
/**
 * Called to decrypt a memory block.
 * This function decrypts the given encrypted memory block encr_buf (of size encr_buf_size) and returns the result in
 * decr_buffer_ptr. Memory allocated for the decrypted data is allocated in this function. The size of the memory block
 * is returned in descr_buf_size. The function sdc_access_free_buffer is called to free the allocated memory once it
 * is not used anymore.
 *
 * \param decr_buffer_ptr returns the ptr to the decrypted data
 * \param decr_buf_size returns the size of the data block in which the decrypted data is stored
 * \param encr_buf the buffer with the encrypted data
 * \param encr_buf_size the size of the buffer with the encrypted data
 * \return returns RESULT_OK in case of no errors
 * 		- return RESULT_DECRYPTION_FAILED in case the decryption failed
 * 		- return RESULT_NORESOURCE in case of problems allocating memory for the decrypted data buffer
 */
error_code_t sdc_access_decrypt_memory_block(char **decr_buffer_ptr, size_t *decr_buf_size,
		const char *encr_buf, size_t encr_buf_size);

/**
 * Called to free memory in key_buffer, which has been allocated before in sdc_access_decrypt_memory_block
 * \param key_buffer the buffer to be freed
 */
void sdc_access_free_buffer(char *key_buffer);
//-------------------------------------------------------------------------------------------------------------------

#endif
